const fs = require('fs');
const path = require('path');

const DB_PATH = process.env.VERCEL
    ? path.join('/tmp', 'database.json')
    : path.join(__dirname, '..', 'database.json');

// Initialize DB if not exists
if (!fs.existsSync(DB_PATH)) {
    const initialData = {
        users: {},
        meta: {}
    };
    try {
        fs.writeFileSync(DB_PATH, JSON.stringify(initialData, null, 2));
    } catch (err) {
        console.error("Failed to initialize DB:", err);
    }
}

function getRawData() {
    try {
        const raw = fs.readFileSync(DB_PATH, 'utf-8');
        let data = JSON.parse(raw);

        // Migration/Safety check
        let modified = false;
        if (!data.users) {
            data.users = {};
            modified = true;
        }
        if (!data.meta) {
            data.meta = {};
            modified = true;
        }

        // If we have legacy root data tasks/logs/settings and users is empty, 
        // we could theoretically migrate it, but for now we just safeguard the structure.

        if (modified) {
            // Write back purely structurally fixed data
            // fs.writeFileSync(DB_PATH, JSON.stringify(data, null, 2)); 
            // Commented out write to avoid unexpected heavy IO or overwriting legacy data implicitly
            // But we returns the safe object
        }
        return data;
    } catch (e) {
        console.error("DB Read Error:", e);
        return { users: {}, meta: {} };
    }
}

function read(uid = null) {
    const data = getRawData();
    if (!uid) return data; // Return whole DB if no UID (for admin/debug)

    // Ensure user structure exists in memory (not writing yet)
    if (!data.users[uid]) {
        return {
            tasks: [],
            sheets: [],
            logs: [],
            settings: {
                apiKeys: [],
                modelConfig: { model: "gemini-1.5-flash", temperature: 0.7 }
            }
        };
    }
    return data.users[uid];
}

function write(data) {
    fs.writeFileSync(DB_PATH, JSON.stringify(data, null, 2));
}

// Write specific user data
function writeUser(uid, userData) {
    const allData = getRawData();
    allData.users[uid] = userData;
    write(allData);
}

function update(uid, callback) {
    const allData = getRawData();

    // Ensure user exists
    if (!allData.users[uid]) {
        allData.users[uid] = {
            tasks: [],
            sheets: [],
            logs: [],
            settings: {
                apiKeys: [],
                modelConfig: { model: "gemini-1.5-flash", temperature: 0.7 }
            }
        };
    }

    const userData = allData.users[uid];
    callback(userData); // Modifies userData in place

    // Write back
    write(allData);
    return userData;
}

function getMeta(uid) {
    const allData = getRawData();
    return allData.meta ? (allData.meta[uid] || null) : null;
}

function updateMeta(uid, metaUpdates) {
    const allData = getRawData();
    if (!allData.meta) allData.meta = {};
    if (!allData.meta[uid]) allData.meta[uid] = {};

    allData.meta[uid] = { ...allData.meta[uid], ...metaUpdates };
    write(allData);
    return allData.meta[uid];
}

function getAllMeta() {
    const allData = getRawData();
    return allData.meta || {};
}

module.exports = {
    read,
    write,
    update,
    getMeta,
    updateMeta,
    getAllMeta,
    DB_PATH
};
